//
// ViewController.m
// Charts Sample
//
// These samples introduce how to create  UIChart object as  a specific Chart type
// that include Pie Graph, Bar Graph, Line Graph, Cylinder,  Area, and other chart
// types on your iOS app layout.
//
// The main idea of the samples is to provide you with a suggestion how to display
// tabular data values as a Chart.  Charts are often the best  way to  demonstrate
// trends in data  and make comparisons between different groups.  Different types
// of graphs are required to  most efficiently and effectively represent different
// types of data.
//
// Also the code example gives a hint how to deal  with source  data when you need
// manipulate and graphically display set of data records in the UIChart interface.
//
// Please read on-line documentation: http://rustemsoft.com/TouchControlsDoc/index.html
//
// Copyright (c) 2015 Rustemsoft LLC. All rights reserved.
//
// ======================================================================================
// Xcode 6.1.1 still has problems related to framework build and referencing.
// We have listed resolutions for couple of possible issues you may encounter:
//
// 1. When you run the Sample project for very first time the build may fail. To fix that
//   remove the reference to TouchControlsTrial.framework and re-reference to the library
//   as following:
// - In the Project Navigator, select the Sample project.
// - Select the target.
// - Select the "General" tab.
// - Open "Embedded Binaries" expander.
// - Click the + button.
// - Select the TouchControlsTrial.framework.
// - Window -> Organizer -> Delete Derived Data and then: Product -> Clean.
//

#import "ViewController.h"

// Device detection. Define size coefficient if it is iPhone or iPad
#define COEFF_PAD ((UI_USER_INTERFACE_IDIOM() == UIUserInterfaceIdiomPad) ? 2.0 : 1.0)

@interface ViewController ()
{
    UIChart *chrt;
    UIButton *btnForward;
    UIButton *btnBackward;
    UILabel *lblSampleTitle;
    UISwitch *swhOrientation;
    UISwitch *swhNegativeSeries;
    UILabel *lblNegative;
    chartTypes currentChartType;
    NSMutableArray *dataItmsSeries0;
    NSMutableArray *dataItmsSeries1;
    NSMutableArray *dataItmsSeries2;
}
@end

@implementation ViewController

- (void)viewDidLoad
{
    [super viewDidLoad];
    // Specify chart type
    currentChartType = pie;

    // Define Chart Layout
    [self defineChart];
    
    // Run Chart
    [chrt DataBind];
    
    // Add Orientation switch to define charts' Vertical/Horizontal orientation
    swhOrientation = [[UISwitch alloc] initWithFrame:CGRectMake(COEFF_PAD * 50, chrt.frame.origin.y + chrt.frame.size.height + COEFF_PAD * 5, COEFF_PAD * 10, COEFF_PAD * 2)];
    swhOrientation.transform = CGAffineTransformMakeScale(COEFF_PAD * .5, COEFF_PAD * .5);
    [swhOrientation addTarget:self action:@selector(swhChanged:) forControlEvents:UIControlEventValueChanged];
    swhOrientation.on = YES;
    swhOrientation.enabled = NO;
    [self.view addSubview:swhOrientation];
    UILabel *lblHorizontal = [[UILabel alloc] initWithFrame:CGRectMake(swhOrientation.frame.origin.x - COEFF_PAD * 34, swhOrientation.frame.origin.y - 2, COEFF_PAD * 100, COEFF_PAD * 20)];
    [lblHorizontal setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7]];
    lblHorizontal.text = @"Horizontal";
    [self.view addSubview:lblHorizontal];
    UILabel *lblVertical = [[UILabel alloc] initWithFrame:CGRectMake(swhOrientation.frame.origin.x + swhOrientation.frame.size.width + 2, swhOrientation.frame.origin.y - 2, COEFF_PAD * 100, COEFF_PAD * 20)];
    [lblVertical setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7]];
    lblVertical.text = @"Vertical";
    [self.view addSubview:lblVertical];

    // Add a switch to add Series with Negative values
    swhNegativeSeries = [[UISwitch alloc] initWithFrame:CGRectMake(COEFF_PAD * 130, chrt.frame.origin.y + chrt.frame.size.height + COEFF_PAD * 5, COEFF_PAD * 10, COEFF_PAD * 2)];
    swhNegativeSeries.transform = CGAffineTransformMakeScale(COEFF_PAD * .5, COEFF_PAD * .5);
    [swhNegativeSeries addTarget:self action:@selector(swhChanged:) forControlEvents:UIControlEventValueChanged];
    swhNegativeSeries.on = NO;
    [self.view addSubview:swhNegativeSeries];
    lblNegative = [[UILabel alloc] initWithFrame:CGRectMake(swhNegativeSeries.frame.origin.x + swhNegativeSeries.frame.size.width + 2, swhNegativeSeries.frame.origin.y - 2, COEFF_PAD * 100, COEFF_PAD * 20)];
    [lblNegative setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7]];
    lblNegative.text = @"non-3D pie";
    [self.view addSubview:lblNegative];
    
    // Add btnBackward button
    btnBackward  = [[UIButton alloc] initWithFrame:CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)];
    [btnBackward setImage:[UIImage imageNamed:@"back.png"] forState:UIControlStateNormal];
    [btnBackward addTarget:self action:@selector(btnBackwardClicked:) forControlEvents:UIControlEventTouchUpInside];
    [btnBackward setAlpha:.5];
    [self.view addSubview:btnBackward];
    
    // Add btnForward button
    btnForward  = [[UIButton alloc] initWithFrame:CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)];
    [btnForward setImage:[UIImage imageNamed:@"forw.png"] forState:UIControlStateNormal];
    [btnForward addTarget:self action:@selector(btnForwardClicked:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:btnForward];
    
    // Define Sample Title label
    lblSampleTitle = [[UILabel alloc] initWithFrame:CGRectMake(COEFF_PAD * 10, COEFF_PAD * 15, COEFF_PAD * 220, COEFF_PAD * 20)];
    lblSampleTitle.text = @"Pie Chart";
    [lblSampleTitle setTextColor:[UIColor purpleColor]];
    [lblSampleTitle setFont:[UIFont fontWithName:@"HelveticaNeue-CondensedBold" size:COEFF_PAD * 16]];
    [self.view addSubview:lblSampleTitle];
}

// Define Chart Layout
- (void)defineChart {
    [chrt removeFromSuperview];
    
    // Create UIChart object
    chrt = [[UIChart alloc] initWithFrame:CGRectMake(COEFF_PAD * 10,  COEFF_PAD * 40, self.view.frame.size.width - COEFF_PAD * 40, self.view.frame.size.height - COEFF_PAD * 190)];
    chrt.delegate = self; // set UIChart's delegate to self (otherwise events will not work)
    
    // Add UIChart object into app's main view layout
    [self.view addSubview:chrt];
    chrt.chartType = currentChartType;
    
    // Define chart's colors
    chrt.backgroundColor = [UIColor colorWithRed:0.01 green:0.01 blue:0.5 alpha:0.1];
    chrt.foregroundColor = [UIColor colorWithRed:0.9 green:0.9 blue:1 alpha:1];

    chrt.orientation = swhOrientation.isOn ? vertical : horizontal;
    
    // Define chart plot area
    chrt.plotArea.backgroundColor = [UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1];
    
    // Define data sources for chart items
    [self dataItemsSeries];
    
    // Draw all kinds of Chart Types
    if (currentChartType == pie) {
        lblSampleTitle.text = @"Pie Chart";

        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        // Define chart's gradient
        chrt.gradientStyle = gradientStyleRadial;
        chrt.gradientRadius = 200;
        chrt.gradientColorTop = [UIColor colorWithRed:0.1 green:0.8 blue:0.1 alpha:0.4];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor clearColor];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.4];
        chrt.gradientDirection = gradientDirectionHorizontal;
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Who did eat more apples?";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 185, COEFF_PAD * 18);
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Jonathan Apple Sauce";
        chrt.xAxisTitleLabel.textColor = [UIColor purpleColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 10];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationMid;
        
        // Set Y axis title
        chrt.yAxisTitleLabel.text = @"McIntosh Apple Pie";
        chrt.yAxisTitleLabel.textColor = [UIColor purpleColor];
        chrt.yAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 10];
        chrt.yAxisTitleLabel.frame = CGRectMake(COEFF_PAD * -83, COEFF_PAD * 160, COEFF_PAD * 185, COEFF_PAD * 18);
        chrt.yAxisTitleVerticalLocation = verticalLocationMid;
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendZone.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        chrt.legendZone.Title.text = @"Persons";
        chrt.legendZone.Title.textColor = [UIColor yellowColor];
        chrt.legendZone.titleVerticalLocation = verticalLocationTop;
        chrt.legendZone.titleHorizontalLocation = horizontalLocationMid;
        chrt.legendZoneHorizontalLocation = horizontalLocationLeft;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        chrt.legendsInterpositions = vertical;
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        
        // 3D Pie or not
        srs0.pie3D = !swhNegativeSeries.on;
        // Define line shadow
        srs0.shape.shadowColor = [UIColor blackColor].CGColor;
        srs0.shape.shadowOpacity = .9;
        srs0.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs0.shape.shadowRadius = 10;
        srs0.shape.shadowPath = srs0.shape.path;
        srs0.shape.lineWidth = COEFF_PAD * .3;
        int i = 0;
        // Fill out chart items collection
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor brownColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.titleVerticalLocation = verticalLocationBottom;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            // Create legend bullet to add into the legend zone
            legend *lgd = [[legend alloc] init];
            lgd.legendLabel.text = [itm objectAtIndex:0];
            lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
            lgd.legendBulletType = legendBulletTypeRect;
            // Specify legend bullet's color
            CGFloat colorHue = i / (CGFloat)(dataItmsSeries0.count);
            lgd.legendBulletFillColor = [UIColor colorWithHue:colorHue saturation:1 brightness:1 alpha:1];
            lgd.legendBulletWidth = COEFF_PAD * 10;
            // Add legend bullet into legend zone
            [chrt.legends addObject:lgd];
            // Add chart item into series
            [srs0.Items addObject:chIt];
            i += 1;
        }
        
        // 3D Pie or not
        srs1.pie3D = !swhNegativeSeries.on;
        srs1.shape.lineWidth = COEFF_PAD * .3;
        srs1.shape.strokeColor = [UIColor redColor].CGColor;
        srs1.pieRadius = 65;
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.pointWidth = 0;
            chIt.layoutStyle = layoutStyleFlat;
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
    } else if (currentChartType == bar) {
        lblSampleTitle.text = @"Bar Chart";

        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleHatch;
        chrt.hatchLineStepHeight = 8;
        chrt.hatchLineWidth = COEFF_PAD * 0.1;
        
        // Also you may define chart's layout as a background image
        //chrt.layoutStyle = layoutStyleImage;
        //chrt.backgroundImageName = @"background.png";
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Apples have been eaten in 3 days";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        
        // Add marked region into chart plot area
        markedRegion *markedRgn = [[markedRegion alloc] init];
        markedRgn.yTop = .5;
        markedRgn.yBottom = .375;
        markedRgn.Title.text = @"Marked Area";
        markedRgn.Title.textColor = [UIColor yellowColor];
        markedRgn.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        // Adjust marked region location
        markedRgn.titleVerticalLocation = verticalLocationTop;
        markedRgn.titleHorizontalLocation = horizontalLocationLeft;
        [chrt.markedRegions addObject:markedRgn];
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Persons";
        chrt.xAxisTitleLabel.textColor = [UIColor purpleColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendsInterpositions = horizontal;
        chrt.legendZoneHorizontalLocation = horizontalLocationRight;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"High value";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_red.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Normal value";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_white.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Low value";
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletImageName = @"flwr_blue.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            // Item Title text
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            // Item Title location
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            // Define item's border
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleSimple;
            chIt.gradientDirection = gradientDirectionDiagonalLeft;
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:0];
            
            // Apply colors depending on max/min/normal values
            if (chIt.value > markedRgn.yTop) { // high value
                chIt.gradientColorTop = [UIColor redColor];
                chIt.gradientColorTopExtra = [UIColor colorWithRed:1 green:0 blue:0 alpha:0.2];
                chIt.gradientColorBottom = [UIColor colorWithRed:1 green:0 blue:0 alpha:0.2];
            } else if (chIt.value < markedRgn.yBottom) { // low value
                chIt.gradientColorTop = [UIColor blueColor];
                chIt.gradientColorTopExtra = [UIColor colorWithRed:0 green:0 blue:0.9 alpha:0.1];
                chIt.gradientColorBottom = [UIColor colorWithRed:0 green:0 blue:0.9 alpha:0.1];
            } else { // normal range
                chIt.gradientColorTop = [UIColor whiteColor];
                chIt.gradientColorTopExtra = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.05];
                chIt.gradientColorBottom = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.05];
            }
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor whiteColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor orangeColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor yellowColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.pointType = pointTypeRect;
            // Define item's border
            chIt.layer.borderWidth = COEFF_PAD * .5;
            // Define item's layout style
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRainbow;
            chIt.gradientDirection = gradientDirectionDiagonalRight;
            // Also you may define item's layout style as an image
            //chIt.backgroundImageName = @"background.png";
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
        
        // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
        chrt.itemBreadth = .6;
    } else if (currentChartType == barStacked) {
        lblSampleTitle.text = @"Stacked Bar Chart";
        
        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleHatch;
        chrt.hatchLineStepHeight = 8;
        chrt.hatchLineWidth = COEFF_PAD * 0.1;
        
        // Also you may define chart's layout as a background image
        //chrt.layoutStyle = layoutStyleImage;
        //chrt.backgroundImageName = @"background.png";
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Apples have been eaten in 3 days";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        
        // Add marked region into chart plot area
        markedRegion *markedRgn = [[markedRegion alloc] init];
        markedRgn.yTop = .5;
        markedRgn.yBottom = .375;
        markedRgn.Title.text = @"Marked Area";
        markedRgn.Title.textColor = [UIColor yellowColor];
        markedRgn.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        // Adjust marked region location
        markedRgn.titleVerticalLocation = verticalLocationTop;
        markedRgn.titleHorizontalLocation = horizontalLocationLeft;
        [chrt.markedRegions addObject:markedRgn];
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Persons";
        chrt.xAxisTitleLabel.textColor = [UIColor purpleColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendsInterpositions = horizontal;
        chrt.legendZoneHorizontalLocation = horizontalLocationRight;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"High value";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_red.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Normal value";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_white.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Low value";
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletImageName = @"flwr_blue.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            // Item Title text
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            // Item Title location
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            // Define item's border
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleSimple;
            chIt.gradientDirection = gradientDirectionDiagonalLeft;
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:0];
            
            // Apply colors depending on max/min/normal values
            if (chIt.value > markedRgn.yTop) { // high value
                chIt.gradientColorTop = [UIColor redColor];
                chIt.gradientColorTopExtra = [UIColor colorWithRed:1 green:0 blue:0 alpha:0.2];
                chIt.gradientColorBottom = [UIColor colorWithRed:1 green:0 blue:0 alpha:0.2];
            } else if (chIt.value < markedRgn.yBottom) { // low value
                chIt.gradientColorTop = [UIColor blueColor];
                chIt.gradientColorTopExtra = [UIColor colorWithRed:0 green:0 blue:0.9 alpha:0.1];
                chIt.gradientColorBottom = [UIColor colorWithRed:0 green:0 blue:0.9 alpha:0.1];
            } else { // normal range
                chIt.gradientColorTop = [UIColor whiteColor];
                chIt.gradientColorTopExtra = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.05];
                chIt.gradientColorBottom = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.05];
            }
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor darkGrayColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor cyanColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor yellowColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.pointType = pointTypeRect;
            // Define item's border
            chIt.layer.borderWidth = COEFF_PAD * .5;
            // Define item's layout style
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRainbow;
            chIt.gradientDirection = gradientDirectionDiagonalRight;
            // Also you may define item's layout style as an image
            //chIt.backgroundImageName = @"background.png";
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
        
        // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
        chrt.itemBreadth = .6;
    } else if (currentChartType == line) {
        lblSampleTitle.text = @"Line Chart";

        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        // Define chart's gradient
        chrt.gradientStyle = gradientStyleRadial;
        chrt.gradientRadius = 200;
        chrt.gradientColorTop = [UIColor colorWithRed:0.1 green:0.8 blue:0.1 alpha:0.4];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor clearColor];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.4];
        chrt.gradientDirection = gradientDirectionHorizontal;
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Prevalence of 3 types of apples across Continents";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 13]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18);
        
        // Add marked region into chart plot area
        markedRegion *markedRgn = [[markedRegion alloc] init];
        markedRgn.yTop = .5;
        markedRgn.yBottom = .3;
        markedRgn.Title.text = @"Marked Area";
        markedRgn.Title.textColor = [UIColor yellowColor];
        markedRgn.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        // Adjust marked region location
        markedRgn.titleVerticalLocation = verticalLocationTop;
        markedRgn.titleHorizontalLocation = horizontalLocationRight;
        [chrt.markedRegions addObject:markedRgn];
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Continents";
        chrt.xAxisTitleLabel.textColor = [UIColor purpleColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendsInterpositions = horizontal;
        chrt.legendZoneHorizontalLocation = horizontalLocationMid;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Rejuvenating Apple";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeTriangle;
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendBulletFillColor = [UIColor whiteColor];
        lgd.legendBulletStrokeLineColor = [UIColor redColor];
        lgd.legendBulletStrokeLineWidth = COEFF_PAD * 2;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Apple of Discord";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeDot;
        lgd.legendBulletFillColor = [UIColor redColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Hesperides Apple";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor blueColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        // Define the series' line shape
        srs0.shape.lineWidth = COEFF_PAD * .3;
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor whiteColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.titleVerticalLocation = verticalLocationBottom;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointType = pointTypeTriangle;
            chIt.pointWidth = COEFF_PAD * 5;
            chIt.pointFillColor = [UIColor whiteColor];
            chIt.pointStrokeLineWidth = COEFF_PAD * 1;
            chIt.pointStrokeLineColor = [UIColor redColor];
            chIt.layoutStyle = layoutStyleFlat;
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:0];
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        // Define the series' line shape
        srs1.shape.lineWidth = COEFF_PAD * 3;
        srs1.shape.strokeColor = [UIColor redColor].CGColor;
        // Define line shadow
        srs1.shape.shadowColor = [UIColor blackColor].CGColor;
        srs1.shape.shadowOpacity = .9;
        srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs1.shape.shadowRadius = 10;
        srs1.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointWidth = 0;
            chIt.pointType = pointTypeDot;
            chIt.pointWidth = COEFF_PAD * 7;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRadial;
            chIt.gradientRadius = COEFF_PAD * 4;
            chIt.gradientColorTop = [UIColor redColor];
            chIt.gradientColorBottomExtra = [UIColor redColor];
            chIt.layer.borderWidth = COEFF_PAD * 1;
            chIt.backgroundColor = [UIColor orangeColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        // Define the series' line shape
        srs2.shape.lineWidth = COEFF_PAD * 3;
        srs2.shape.strokeColor = [UIColor blueColor].CGColor;
        // Define line shadow
        srs2.shape.shadowColor = [UIColor blackColor].CGColor;
        srs2.shape.shadowOpacity = .9;
        srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs2.shape.shadowRadius = 10;
        srs2.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor blueColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointType = pointTypeRect;
            chIt.pointWidth = COEFF_PAD * 7;
            chIt.layer.borderWidth = COEFF_PAD * 1;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRadial;
            chIt.gradientRadius = COEFF_PAD * 3;
            chIt.backgroundColor = [UIColor brownColor];
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
    } else if (currentChartType == area) {
        lblSampleTitle.text = @"Area Chart";
        
        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        // Define chart's gradient
        chrt.gradientStyle = gradientStyleRadial;
        chrt.gradientRadius = 200;
        chrt.gradientColorTop = [UIColor colorWithRed:0.1 green:0.8 blue:0.1 alpha:0.4];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor clearColor];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.4];
        chrt.gradientDirection = gradientDirectionHorizontal;
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Apple Comparison Chart";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18);
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Recommended Usage";
        chrt.xAxisTitleLabel.textColor = [UIColor lightTextColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        chrt.xAxisTitleYshift = 11;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        chrt.xAxisLabelsShowDefaultOrderNumbers = NO;
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor brownColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.3;
        chrt.legendsInterpositions = horizontal;
        chrt.legendZoneHorizontalLocation = horizontalLocationLeft;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"McIntosh";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor whiteColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Braeburn";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor redColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Stayman";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor blueColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor whiteColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleSimple;
            chIt.gradientColorTop = [UIColor whiteColor];
            chIt.gradientColorTopExtra = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.2];
            chIt.gradientColorBottom = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.2];
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        // Define the series' line shape
        srs1.shape.lineWidth = COEFF_PAD * 3;
        srs1.shape.strokeColor = [UIColor redColor].CGColor;
        // Define line shadow
        srs1.shape.shadowColor = [UIColor blackColor].CGColor;
        srs1.shape.shadowOpacity = .9;
        srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs1.shape.shadowRadius = 10;
        srs1.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointWidth = 0;
            chIt.pointType = pointTypeDot;
            chIt.pointWidth = COEFF_PAD * 7;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRadial;
            chIt.gradientRadius = COEFF_PAD * 4;
            chIt.gradientColorTop = [UIColor redColor];
            chIt.gradientColorBottomExtra = [UIColor redColor];
            chIt.layer.borderWidth = COEFF_PAD * 1;
            chIt.backgroundColor = [UIColor orangeColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        // Define the series' line shape
        srs2.shape.lineWidth = COEFF_PAD * 3;
        srs2.shape.strokeColor = [UIColor blueColor].CGColor;
        // Define line shadow
        srs2.shape.shadowColor = [UIColor blackColor].CGColor;
        srs2.shape.shadowOpacity = .9;
        srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs2.shape.shadowRadius = 10;
        srs2.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor blueColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointType = pointTypeRect;
            chIt.pointWidth = COEFF_PAD * 7;
            chIt.layer.borderWidth = COEFF_PAD * 1;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRadial;
            chIt.gradientRadius = COEFF_PAD * 3;
            chIt.backgroundColor = [UIColor brownColor];
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:0];
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
    } else if (currentChartType == areaStacked) {
        lblSampleTitle.text = @"Stacked Area Chart";
        
        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        // Define chart's gradient
        chrt.gradientStyle = gradientStyleRadial;
        chrt.gradientRadius = 200;
        chrt.gradientColorTop = [UIColor colorWithRed:0.1 green:0.8 blue:0.1 alpha:0.4];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor clearColor];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.4];
        chrt.gradientDirection = gradientDirectionHorizontal;
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Apple Comparison Chart";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18);
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Recommended Usage";
        chrt.xAxisTitleLabel.textColor = [UIColor lightTextColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        chrt.xAxisTitleYshift = 11;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        chrt.xAxisLabelsShowDefaultOrderNumbers = NO;
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor brownColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.3;
        chrt.legendsInterpositions = horizontal;
        chrt.legendZoneHorizontalLocation = horizontalLocationLeft;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"McIntosh";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor whiteColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Braeburn";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor redColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Stayman";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor blueColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        // Define the series' line shape
        srs0.shape.lineWidth = COEFF_PAD * .3;
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor whiteColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.titleVerticalLocation = verticalLocationBottom;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleSimple;
            chIt.gradientColorTop = [UIColor whiteColor];
            chIt.gradientColorTopExtra = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.9];
            chIt.gradientColorBottom = [UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.2];
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:0];
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        // Define the series' line shape
        srs1.shape.lineWidth = COEFF_PAD * 3;
        srs1.shape.strokeColor = [UIColor redColor].CGColor;
        // Define line shadow
        srs1.shape.shadowColor = [UIColor blackColor].CGColor;
        srs1.shape.shadowOpacity = .9;
        srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs1.shape.shadowRadius = 10;
        srs1.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointWidth = 0;
            chIt.pointType = pointTypeDot;
            chIt.pointWidth = COEFF_PAD * 7;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRadial;
            chIt.gradientRadius = COEFF_PAD * 4;
            chIt.gradientColorTop = [UIColor redColor];
            chIt.gradientColorBottomExtra = [UIColor redColor];
            chIt.layer.borderWidth = COEFF_PAD * 1;
            chIt.backgroundColor = [UIColor orangeColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        // Define the series' line shape
        srs2.shape.lineWidth = COEFF_PAD * 3;
        srs2.shape.strokeColor = [UIColor blueColor].CGColor;
        // Define line shadow
        srs2.shape.shadowColor = [UIColor blackColor].CGColor;
        srs2.shape.shadowOpacity = .9;
        srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs2.shape.shadowRadius = 10;
        srs2.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor blueColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointType = pointTypeRect;
            chIt.pointWidth = COEFF_PAD * 7;
            chIt.layer.borderWidth = COEFF_PAD * 1;
            chIt.layoutStyle = layoutStyleGradient;
            chIt.gradientStyle = gradientStyleRadial;
            chIt.gradientRadius = COEFF_PAD * 3;
            chIt.backgroundColor = [UIColor brownColor];
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
    } else if (currentChartType == point) {
        lblSampleTitle.text = @"Point Chart";
        
        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        // Define chart's gradient
        chrt.gradientStyle = gradientStyleRadial;
        chrt.gradientRadius = 200;
        chrt.gradientColorTop = [UIColor colorWithRed:0.1 green:0.8 blue:0.1 alpha:0.4];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor clearColor];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.4];
        chrt.gradientDirection = gradientDirectionHorizontal;
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Prevalence of 3 types of apples across Continents";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 13]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor whiteColor]];
        chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18);
        
        // Add marked region into chart plot area
        markedRegion *markedRgn = [[markedRegion alloc] init];
        markedRgn.yTop = .5;
        markedRgn.yBottom = .3;
        markedRgn.Title.text = @"Marked Area";
        markedRgn.Title.textColor = [UIColor yellowColor];
        markedRgn.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        // Adjust marked region location
        markedRgn.titleVerticalLocation = verticalLocationTop;
        markedRgn.titleHorizontalLocation = horizontalLocationRight;
        [chrt.markedRegions addObject:markedRgn];
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Continents";
        chrt.xAxisTitleLabel.textColor = [UIColor purpleColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendsInterpositions = horizontal;
        chrt.legendZoneHorizontalLocation = horizontalLocationMid;
        chrt.legendZoneVerticalLocation = verticalLocationBottom;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Rejuvenating Apple";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeTriangle;
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendBulletFillColor = [UIColor whiteColor];
        lgd.legendBulletStrokeLineColor = [UIColor redColor];
        lgd.legendBulletStrokeLineWidth = COEFF_PAD * 2;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Apple of Discord";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeDot;
        lgd.legendBulletFillColor = [UIColor redColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Hesperides Apple";
        lgd.legendLabel.textColor = [UIColor whiteColor];
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeRect;
        lgd.legendBulletFillColor = [UIColor blueColor];
        lgd.legendBulletWidth = COEFF_PAD * 12;
        lgd.legendMargin = COEFF_PAD * 3;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        // Define the series' line shape
        srs0.shape.lineWidth = COEFF_PAD * .3;
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor whiteColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.titleVerticalLocation = verticalLocationBottom;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointType = pointTypeTriangle;
            chIt.pointWidth = COEFF_PAD * 9;
            chIt.pointFillColor = [UIColor whiteColor];
            chIt.pointStrokeLineWidth = COEFF_PAD * .2;
            chIt.pointStrokeLineColor = [UIColor redColor];
            chIt.layoutStyle = layoutStyleFlat;
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:0];
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        // Define the series' line shape
        srs1.shape.lineWidth = COEFF_PAD * 3;
        srs1.shape.strokeColor = [UIColor redColor].CGColor;
        // Define line shadow
        srs1.shape.shadowColor = [UIColor blackColor].CGColor;
        srs1.shape.shadowOpacity = .9;
        srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs1.shape.shadowRadius = 10;
        srs1.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointWidth = 0;
            chIt.pointType = pointTypeDot;
            chIt.pointWidth = COEFF_PAD * 9;
            chIt.pointFillColor = [UIColor redColor];
            chIt.layoutStyle = layoutStyleFlat;
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        // Define the series' line shape
        srs2.shape.lineWidth = COEFF_PAD * 3;
        srs2.shape.strokeColor = [UIColor blueColor].CGColor;
        // Define line shadow
        srs2.shape.shadowColor = [UIColor blackColor].CGColor;
        srs2.shape.shadowOpacity = .9;
        srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10);
        srs2.shape.shadowRadius = 10;
        srs2.shape.shadowPath = srs0.shape.path;
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor blueColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleVerticalLocation = verticalLocationTop;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.pointType = pointTypeRect;
            chIt.pointWidth = COEFF_PAD * 9;
            chIt.pointFillColor = [UIColor blueColor];
            chIt.layoutStyle = layoutStyleFlat;
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
    } else if (currentChartType == cylinder) {
        lblSampleTitle.text = @"Cylinder Chart";

        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        chrt.gradientStyle = gradientStyleCross;
        chrt.gradientColorTop = [UIColor colorWithRed:0.2 green:0.1 blue:0.1 alpha:0.1];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor colorWithRed:0.2 green:0.1 blue:0.1 alpha:0.1];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.2 green:0.05 blue:0.05 alpha:0.1];
        chrt.gradientDirection = gradientDirectionDiagonalLeft;

        // Also you may define chart's layout as a background image
        //chrt.layoutStyle = layoutStyleImage;
        //chrt.backgroundImageName = @"background.png";
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Some Apples are Sweet, some tart...";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor lightTextColor]];
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Apples";
        chrt.xAxisTitleLabel.textColor = [UIColor orangeColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        chrt.xAxisTitleLabelRotationAngle = -10;
        chrt.xAxisTitleXshift = 11;
        chrt.xAxisTitleYshift = 25;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendsInterpositions = vertical;
        chrt.legendZoneHorizontalLocation = horizontalLocationRight;
        chrt.legendZoneVerticalLocation = verticalLocationTop;
        chrt.legendZone.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        chrt.legendZone.Title.text = @"Best for ";
        chrt.legendZone.Title.textColor = [UIColor yellowColor];
        chrt.legendZone.titleVerticalLocation = verticalLocationMid;
        chrt.legendZone.titleHorizontalLocation = horizontalLocationLeft;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Cooking";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_red.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Baking";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_white.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Eating Fresh";
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletImageName = @"flwr_blue.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            // Item Title text
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            // Item Title location
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            // Define item's border
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor redColor];
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:2];
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor darkGrayColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor whiteColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor yellowColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.pointType = pointTypeRect;
            // Define item's border
            chIt.layer.borderWidth = COEFF_PAD * .5;
            // Define item's layout style
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor blueColor];
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
        
        // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
        chrt.itemBreadth = .6;
    } else if (currentChartType == cylinderStacked) {
        lblSampleTitle.text = @"Stacked Cylinder Chart";

        // Define chart's Layout style
        chrt.layoutStyle = layoutStyleGradient;
        chrt.gradientStyle = gradientStyleSquare;
        chrt.gradientColorTop = [UIColor colorWithRed:0.2 green:0.1 blue:0.1 alpha:0.1];
        chrt.gradientColorTopExtra = [UIColor clearColor];
        chrt.gradientColorBottom = [UIColor colorWithRed:0.2 green:0.1 blue:0.1 alpha:0.1];
        chrt.gradientColorBottomExtra = [UIColor colorWithRed:0.2 green:0.05 blue:0.05 alpha:0.1];
        chrt.gradientDirection = gradientDirectionHorizontal;
        
        // Assign chart's Title
        chrt.chartTitle.text = @"Some Apples are Sweet, some tart...";
        [chrt.chartTitle setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 16]];
        [chrt.chartTitle setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.8 alpha:0.1]];
        [chrt.chartTitle setTextColor:[UIColor lightTextColor]];
        
        // Set X axis title
        chrt.xAxisTitleLabel.text = @"Apples";
        chrt.xAxisTitleLabel.textColor = [UIColor orangeColor];
        chrt.xAxisTitleLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 12];
        chrt.xAxisTitleHorizontalLocation = horizontalLocationRight;
        chrt.xAxisTitleLabelRotationAngle = -10;
        chrt.xAxisTitleXshift = 11;
        chrt.xAxisTitleYshift = 25;
        // X axis labels
        chrt.xAxisLinesHidden = NO;
        chrt.xAxisLabelsHidden = NO;
        chrt.xAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.xAxisLabelsTextColor = [UIColor yellowColor];
        // Y axis labels
        chrt.yAxisLabelsFont = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
        chrt.yAxisLabelsTextColor = [UIColor blueColor];
        
        // Draw Legend Zone
        chrt.legendZone.layer.borderColor = [UIColor yellowColor].CGColor;
        chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5;
        chrt.legendsInterpositions = vertical;
        chrt.legendZoneHorizontalLocation = horizontalLocationRight;
        chrt.legendZoneVerticalLocation = verticalLocationTop;
        chrt.legendZone.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 8];
        chrt.legendZone.Title.text = @"Best for ";
        chrt.legendZone.Title.textColor = [UIColor yellowColor];
        chrt.legendZone.titleVerticalLocation = verticalLocationMid;
        chrt.legendZone.titleHorizontalLocation = horizontalLocationLeft;
        
        // Create legend items to add into the legend zone
        legend *lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Cooking";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_red.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Baking";
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendBulletImageName = @"flwr_white.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        lgd = [[legend alloc] init];
        lgd.legendLabel.text = @"Eating Fresh";
        lgd.legendBulletType = legendBulletTypeImage;
        lgd.legendLabel.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 7];
        lgd.legendBulletImageName = @"flwr_blue.gif";
        lgd.legendBulletWidth = COEFF_PAD * 12;
        [chrt.legends addObject:lgd];
        
        // Define chart series
        ChartSeries *srs0 = [[ChartSeries alloc] init];
        ChartSeries *srs1 = [[ChartSeries alloc] init];
        ChartSeries *srs2 = [[ChartSeries alloc] init];
        
        // Fill out chart items of 1st series
        for (NSMutableArray *itm in dataItmsSeries0) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            // Item Title text
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor blackColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            // Item Title location
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.titleHorizontalLocation = horizontalLocationMid;
            // Define item's border
            chIt.layer.borderColor = [UIColor whiteColor].CGColor;
            chIt.layer.borderWidth = COEFF_PAD * 0.3;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor redColor];
            // X axis label text
            chIt.xAxisLabel.text = [itm objectAtIndex:2];
            // Add chart item into series
            [srs0.Items addObject:chIt];
        }
        
        // Fill out chart items of 2nd series
        for (NSMutableArray *itm in dataItmsSeries1) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue];
            chIt.Title.text = [itm objectAtIndex:1];
            chIt.Title.textColor = [UIColor darkGrayColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor whiteColor];
            // Add chart item into series
            [srs1.Items addObject:chIt];
        }
        
        // Fill out chart items of 3rd series
        for (NSMutableArray *itm in dataItmsSeries2) {
            ChartItem *chIt = [[ChartItem alloc] init];
            chIt.value = [[itm objectAtIndex:1] floatValue] * (swhNegativeSeries.on ? -1 : 1);
            chIt.Title.text = [NSString stringWithFormat:@"%.2f", chIt.value];
            chIt.Title.textColor = [UIColor yellowColor];
            chIt.Title.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6];
            chIt.titleHorizontalLocation = horizontalLocationMid;
            chIt.titleVerticalLocation = verticalLocationMid;
            chIt.pointType = pointTypeRect;
            // Define item's border
            chIt.layer.borderWidth = COEFF_PAD * .5;
            // Define item's layout style
            chIt.layoutStyle = layoutStyleFlat;
            chIt.backgroundColor = [UIColor blueColor];
            // Add chart item into series
            [srs2.Items addObject:chIt];
        }
        // Add series into chart
        [chrt.series addObject:srs0];
        [chrt.series addObject:srs1];
        [chrt.series addObject:srs2];
        
        // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
        chrt.itemBreadth = .9;
    }
}

// Define data source for chart items in 3 Series
- (void)dataItemsSeries {
    // Data source for 1st Series
    dataItmsSeries0 = [[NSMutableArray alloc] init];
    NSMutableArray *itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Anthony"];
    [itm addObject:@".40"];
    [itm addObject:@"Ambrosia"];
    [dataItmsSeries0 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Mary"];
    [itm addObject:@".5"];
    [itm addObject:@"Pink Lady"];
    [dataItmsSeries0 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Russ"];
    [itm addObject:@".45"];
    [itm addObject:@"McIntosh"];
    [dataItmsSeries0 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Maxwell"];
    [itm addObject:@".35"];
    [itm addObject:@"Fuji"];
    [dataItmsSeries0 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Tom"];
    [itm addObject:@".7"];
    [itm addObject:@"Gala"];
    [dataItmsSeries0 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Michell"];
    [itm addObject:@".60"];
    [itm addObject:@"Golden Delicious"];
    [dataItmsSeries0 addObject:itm];

    // Data source for 2nd Series
    dataItmsSeries1 = [[NSMutableArray alloc] init];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Hyperborea"];
    [itm addObject:@".50"];
    [dataItmsSeries1 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Atlantis"];
    [itm addObject:@".2"];
    [dataItmsSeries1 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Arktida"];
    [itm addObject:@".65"];
    [dataItmsSeries1 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Lemuria"];
    [itm addObject:@".45"];
    [dataItmsSeries1 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Rutas"];
    [itm addObject:@".8"];
    [dataItmsSeries1 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"Pacifis"];
    [itm addObject:@".5"];
    [dataItmsSeries1 addObject:itm];
    
    // Data source for 3rd Series
    dataItmsSeries2 = [[NSMutableArray alloc] init];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"RAW"];
    [itm addObject:@".80"];
    [dataItmsSeries2 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"PIES"];
    [itm addObject:@".7"];
    [dataItmsSeries2 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"BAKING"];
    [itm addObject:@".55"];
    [dataItmsSeries2 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"SAUCE"];
    [itm addObject:@".6"];
    [dataItmsSeries2 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"JUICE"];
    [itm addObject:@".9"];
    [dataItmsSeries2 addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"WINE"];
    [itm addObject:@".8"];
    [dataItmsSeries2 addObject:itm];
}

// rotate iPhone/iPad
- (void)willRotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration {
    // Define main screen width and height
    CGFloat mainWidth = ((toInterfaceOrientation == UIInterfaceOrientationLandscapeLeft || toInterfaceOrientation == UIInterfaceOrientationLandscapeRight) ? self.view.frame.size.height : self.view.frame.size.width);
    CGFloat mainHeight = ((toInterfaceOrientation == UIInterfaceOrientationLandscapeLeft || toInterfaceOrientation == UIInterfaceOrientationLandscapeRight) ? self.view.frame.size.width : self.view.frame.size.height);
    btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
    btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
}

// Click btnForward button
- (void)btnForwardClicked:(UIButton *)btn {
    // Set next chart type
    currentChartType += 1;
    btnBackward.alpha = 1;
    if (currentChartType == 9) {
        currentChartType = 0;
        btnBackward.alpha = .5;
    }
    swhOrientation.enabled = !(currentChartType == 0);
    lblNegative.text = (currentChartType == 0) ? @"non-3D Pie" : @"Negative Series";
    // Set up UIDataGrid object's layout to show TextView Column Sample
    chrt.chartType = currentChartType;

    // Define Chart Layout
    [self defineChart];

    // Run Chart
    [chrt DataBind];
    btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
    btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
}

// Click btnBackward button
- (void)btnBackwardClicked:(UIButton *)btn {
    if (currentChartType == 0) return;
    // Set next chart type
    currentChartType -= 1;
    if (currentChartType == 0) btnBackward.alpha = .5;
    swhOrientation.enabled = !(currentChartType == 0);
    lblNegative.text = (currentChartType == 0) ? @"non-3D Pie" : @"Negative Series";
    // Set up UIDataGrid object's layout to show TextView Column Sample
    chrt.chartType = currentChartType;
    
    // Define Chart Layout
    [self defineChart];

    // Run Chart
    [chrt DataBind];
    btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
    btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
}

// Change Chart orientation or add/delete Negative Series
- (void)swhChanged:(UISwitch *)swch {
    // Define Chart Layout
    [self defineChart];
    
    // Run Chart
    [chrt DataBind];
    btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
    btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15);
}

// Event-handling procedure that fires when the chart is tapped
- (void)chartTapped:(UIChart *)chart {
    // Show a message
    [self msgbox:@"The chartTapped event has been implemented." title:@"Chart is Tapped!"];
    return;
}

// Show message box
- (void)msgbox:(NSString *)msg title:(NSString *)title {
    UIAlertView *alert = [[UIAlertView alloc] initWithTitle:title message:msg delegate:self cancelButtonTitle:@"Ok" otherButtonTitles:nil, nil];
	[alert show];
}

@end





















